<?php
/**
 * Copyright (c) 2014 Michael Dowling, https://github.com/mtdowling
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

namespace DTS\eBaySDK\JmesPath;

/**
 * Uses an external tree visitor to interpret an AST.
 */
class AstRuntime {
	private $parser;
	private $interpreter;
	private $cache = [];
	private $cachedCount = 0;

	public function __construct(
		Parser $parser = null,
		callable $fnDispatcher = null
	) {
		$fnDispatcher      = $fnDispatcher ?: FnDispatcher::getInstance();
		$this->interpreter = new TreeInterpreter( $fnDispatcher );
		$this->parser      = $parser ?: new Parser();
	}

	/**
	 * Returns data from the provided input that matches a given JMESPath
	 * expression.
	 *
	 * @param string $expression JMESPath expression to evaluate
	 * @param mixed $data Data to search. This data should be data that
	 *                           is similar to data returned from json_decode
	 *                           using associative arrays rather than objects.
	 *
	 * @return mixed|null Returns the matching data or null
	 */
	public function __invoke( $expression, $data ) {
		if ( ! isset( $this->cache[ $expression ] ) ) {
			// Clear the AST cache when it hits 1024 entries
			if ( ++ $this->cachedCount > 1024 ) {
				$this->cache       = [];
				$this->cachedCount = 0;
			}
			$this->cache[ $expression ] = $this->parser->parse( $expression );
		}

		return $this->interpreter->visit( $this->cache[ $expression ], $data );
	}
}
