<?php

/**
 * Api Methods Interface.
 *
 * A protocol specific class that implements this interface must implement all
 * methods defined below. This interface contains all methods supported by
 * the zanox Api.
 *
 * Supported Version: PHP >= 5.0
 *
 * @author      Thomas Nicolai (thomas.nicolai@sociomantic.com)
 * @author      Lars Kirchhoff (lars.kirchhoff@sociomantic.com)
 *
 * @see         http://wiki.zanox.com/en/Web_Services
 * @see         http://apps.zanox.com
 *
 * @package     ApiClient
 * @version     2009-09-01
 * @copyright   Copyright (c) 2007-2009 zanox.de AG
 */
interface IMethods {

	/**
	 * Get a single product.
	 *
	 * @param      string $zupId product id hash
	 * @param      int $adspaceId adspace id (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            single product item or false
	 */
	public function getProduct( $zupId, $adspaceId = null );


	/**
	 * Get product categories.
	 *
	 * @param      int $rootCategory category id (optional)
	 * @param      bool $includeChilds include child nodes (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            single product item or false
	 */
	public function getProductCategories(
		$rootCategory = 0,
		$includeChilds = false
	);


	/**
	 * Get products by advertiser program.
	 *
	 * @param      int $programId advertiser program id
	 * @param      int $adspaceId adspace id (optional)
	 * @param      string $modifiedDate last modification date (optional)
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            product result set or false
	 */
	public function getProducts(
		$programId, $adspaceId = null,
		$modifiedDate = null, $page = 0, $items = 10
	);


	/**
	 * Search for products.
	 *
	 * @param      string $query search string
	 * @param      string $searchType search type (optional)
	 *                                         (contextual or phrase)
	 * @param      string $ip products with sales region
	 *                                         within the region of the ip
	 *                                         address (optional)
	 * @param      string $region limit search to region (optional)
	 * @param      int $categoryId limit search to categorys (optional)
	 * @param      array $programId limit search to program list of
	 *                                         programs (optional)
	 * @param      boolean $hasImages products with images (optional)
	 * @param      float $minPrice minimum price (optional)
	 * @param      float $maxPrice maximum price (optional)
	 * @param      int $adspaceId adspace id (optional)
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            list of products or false
	 */
	public function searchProducts(
		$query, $searchType = 'phrase', $ip = null,
		$region = null, $categoryId = null, $programId = array(),
		$hasImages = true, $minPrice = 0, $maxPrice = null, $adspaceId = null,
		$page = 0, $items = 10
	);


	/**
	 * Retrieve a single zanox advertiser program item.
	 *
	 * @param      int $programId id of program to retrieve
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            program item or false
	 */
	public function getProgram( $programId );


	/**
	 * Get advertiser program categories.
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            category result set or false
	 */
	public function getProgramCategories();


	/**
	 * Get advertiser program applications by adspace.
	 *
	 * @param      int $adspaceId advertising space id
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            program result set or false
	 */
	public function getProgramsByAdspace( $adspaceId, $page = 0, $items = 10 );


	/**
	 * Search zanox advertiser programs.
	 *
	 * @param      string $query search string
	 * @param      string $startDate program start date (optional)
	 * @param      string $partnerShip partnership status (optional)
	 *                                         (direct or indirect)
	 * @param      boolean $hasProducts program has product data
	 * @param      string $ip programs in region of ip address
	 * @param      string $region program region
	 * @param      string $categoryId program category id
	 * @param      int $page page of result set
	 * @param      int $items items per page
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            programs result set or false
	 */
	public function searchPrograms(
		$query, $startDate = null,
		$partnerShip = null, $hasProducts = false, $ip = null,
		$region = null, $categoryId = null, $page = 0, $items = 10
	);


	/**
	 * Create program application for a given adspace.
	 *
	 * @param      int $programId advertiser program id
	 * @param      int $adspaceId advertising space id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     boolean                    true or false
	 */
	public function createProgramApplication( $programId, $adspaceId );


	/**
	 * Delete program application.
	 *
	 * @param      int $programId advertiser program id
	 * @param      int $adspaceId advertising space id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     boolean                     true or false
	 */
	public function deleteProgramApplication( $programId, $adspaceId );


	/**
	 * Get a single admedium.
	 *
	 * @param      int $admediumId advertising medium id
	 * @param      int $adspaceId advertising space id (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            single product item or false
	 */
	public function getAdmedium( $admediumId, $adspaceId = null );


	/**
	 * Get admedium categories.
	 *
	 * @param      int $programId program admedium categories
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            list of admedium categories
	 */
	public function GetAdmediumCategories( $programId );


	/**
	 * Retrieve all advertising media items.
	 *
	 * @param      int $programId advertiser program id (optional)
	 * @param      string $ip ip address (geo feature) (optional)
	 * @param      string $region limit search to region (optional)
	 * @param      string $format admedia format (optional)
	 * @param      string $partnerShip partnership status (optional)
	 *                                         (direct or indirect)
	 * @param      string $purpose purpose of admedia (optional)
	 *                                         (startPage, productDeeplink,
	 *                                         categoryDeeplink, searchDeeplink)
	 * @param      string $admediumType type of admedium (optional)
	 *                                         (html, script, lookatMedia, image,
	 *                                         imageText, text)
	 * @param      int $categoryId admedium category id (optional)
	 * @param      int $adspaceId adspace id (optional)
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   nosignature
	 *
	 * @return     object or string            admedia result set or false
	 */
	public function getAdmedia(
		$programId = null, $ip = null, $region = null,
		$format = null, $partnerShip = null, $purpose = null,
		$admediumType = null, $categoryId = null, $adspaceId = null, $page = 0,
		$items = 10
	);


	/**
	 * Returns a single advertising spaces.
	 *
	 * @param      int $adspaceId advertising space id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            adspace item or false
	 */
	public function getAdspace( $adspaceId );


	/**
	 * Returns all advertising spaces.
	 *
	 * @param      int $page result set page (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            adspaces result set or false
	 */
	public function getAdspaces( $page = 0, $items = 10 );


	/**
	 * Create advertising space (signature).
	 *
	 * @param      string $name adspace name
	 * @param      string $lang language of adspace (e.g. en)
	 * @param      string $url url of adspace
	 * @param      string $contact contact address (email)
	 * @param      string $description description of adspace
	 * @param      string $adspaceType adspace typ (website, email or searchengine)
	 * @param      array $scope adspace scope (private or business)
	 * @param      int $visitors adspace monthly visitors
	 * @param      int $impressions adspace monthly page impressions
	 * @param      string $keywords keywords for adspace (optional)
	 * @param      array $regions adspace customer regions (optional)
	 * @param      array $categories adspace categories (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            adspace item or false
	 */
	public function createAdspace(
		$name, $lang, $url, $contact, $description,
		$adspaceType, $scope, $visitors, $impressions, $keywords = null,
		$regions = array(), $categories = array()
	);


	/**
	 * Update advertising space.
	 *
	 * @param      int $adspaceId adspace id
	 * @param      string $name adspace name
	 * @param      string $lang language of adspace (e.g. en)
	 * @param      string $url url of adspace
	 * @param      string $contact contact address (email)
	 * @param      string $description description of adspace
	 * @param      string $adspaceType adspace typ (website, email or searchengine)
	 * @param      array $scope adspace scope (private or business)
	 * @param      int $visitors adspace monthly visitors
	 * @param      int $impressions adspace monthly page impressions
	 * @param      string $keywords keywords for adspace (optional)
	 * @param      array $regions adspace customer regions (optional)
	 * @param      array $categories adspace categories (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            adspace item or false
	 */
	public function updateAdspace(
		$adspaceId, $name, $lang, $url, $contact,
		$description, $adspaceType, $scope, $visitors, $impressions,
		$keywords = null, $regions = array(), $categories = array()
	);


	/**
	 * Removes advertising space.
	 *
	 * @param      int $adspaceId advertising space id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     boolean                     true on success
	 */
	public function deleteAdspace( $adspaceId );


	/**
	 * Return zanox user profile.
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            profile item
	 */
	public function getProfile();


	/**
	 * Update zanox user profile.
	 *
	 * @param      array $profileId user profile id
	 * @param      array $firstName first name
	 * @param      array $lastName last name
	 * @param      array $email email address
	 * @param      array $country country or residence
	 * @param      array $street1 street 1
	 * @param      array $street2 street 2 (optional)
	 * @param      array $city city
	 * @param      array $company name of company (optional)
	 * @param      array $phone phone number (optional)
	 * @param      array $mobile mobile number (optional)
	 * @param      array $fax fax number (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     boolean                     true on success
	 */
	public function updateProfile(
		$profileId, $firstName, $lastName, $email,
		$country, $street1, $street2 = null, $city, $zipcode, $company = null,
		$phone = null, $mobile = null, $fax = null
	);


	/**
	 * Get back accounts.
	 *
	 * @param      int $page result set page (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            account balances result set or
	 *                                         false
	 */
	public function getBankAccounts( $page = 0, $items = 10 );


	/**
	 * Get single back account.
	 *
	 * @param      int $bankAccountId result set page
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            account balances result set or
	 *                                         false
	 */
	public function getBankAccount( $bankAccountId );


	/**
	 * Get account balance
	 *
	 * @param      int $currency currence code of balance account
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            payment item or false
	 */
	public function getBalance( $currency );


	/**
	 * Get currency account balances.
	 *
	 * @param      int $page result set page (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            account balances result set or
	 *                                         false
	 */
	public function getBalances( $page = 0, $items = 10 );


	/**
	 * Get payment transactions of the current zanox account.
	 *
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            payments result set or false
	 */
	public function getPayments( $page = 0, $items = 10 );


	/**
	 * Get a single payment item.
	 *
	 * @param      int $paymentId payment item id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            payment item or false
	 */
	public function getPayment( $paymentId );


	/**
	 * Get basic sales/leads report.
	 *
	 * @param      string $fromDate report start date
	 * @param      string $toDate report end date
	 * @param      string $dateType type of date to filter by (optional)
	 *                                         (clickDate, trackingDate,
	 *                                         modifiedDate)
	 * @param      string $currency currency (optional)
	 * @param      int $programId program id (optional)
	 * @param      int $admediumId admedium id (optional)
	 * @param      int $admediumFormat admedium format id (optional)
	 * @param      int $adspaceId adspace id (optional)
	 * @param      string $reviewState filter by review status (optional)
	 *                                         (confirmed, open, rejected or
	 *                                         approved)
	 * @param      string $groupBy group report by option (optional)
	 *                                         (country, region, city, currency,
	 *                                         admedium, program, adspace,
	 *                                         linkFormat, reviewState,
	 *                                         trackingCategory, month, day,
	 *                                         hour, year, dayOfWeek)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            payment item or false
	 */
	public function getReportBasic(
		$fromDate, $toDate, $dateType = null,
		$currency = null, $programId = null, $admediumId = null,
		$admediumFormat = null, $adspaceId = null, $reviewState = null,
		$groupBy = null
	);


	/**
	 * Get sales report.
	 *
	 * @param      string $date date of sales
	 * @param      string $dateType type of date to filter by (optional)
	 *                                         (clickDate, trackingDate,
	 *                                         modifiedDate)
	 * @param      int $programId filter by program id (optional)
	 * @param      int $adspaceId filter by adspace id (optional)
	 * @param      array $reviewState filter by review status (optional)
	 *                                         (confirmed, open, rejected or
	 *                                         approved)
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            sales result set or false
	 */
	public function getSales(
		$date, $dateType = null, $programId = null,
		$adspaceId = null, $reviewState = null, $page = 0, $items = 10
	);


	/**
	 * Get single sale item.
	 *
	 * @param      int $saleId sale id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            sales result set or false
	 */
	public function getSale( $saleId );


	/**
	 * Get leads report.
	 *
	 * @param      string $date date of sales
	 * @param      string $dateType type of date to filter by (optional)
	 *                                         (clickDate, trackingDate,
	 *                                         modifiedDate)
	 * @param      int $programId filter by program id (optional)
	 * @param      int $adspaceId filter by adspace id (optional)
	 * @param      array $reviewState filter by review status (optional)
	 *                                         (confirmed, open, rejected or
	 *                                         approved)
	 * @param      int $page page of result set (optional)
	 * @param      int $items items per page (optional)
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            sales result set or false
	 */
	public function getLeads(
		$date, $dateType = null, $programId = null,
		$adspaceId = null, $reviewState = null, $page = 0, $items = 10
	);


	/**
	 * Get single sale item.
	 *
	 * @param      int $leadId lead id
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object or string            sales result set or false
	 */
	public function getLead( $leadId );


	/**
	 * Returns new OAuth user session
	 *
	 * @param      string $authToken authentication token
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     object                      user session
	 */
	public function getSession( $authToken );


	/**
	 * Closes OAuth user session
	 *
	 * @param      string $connectId connect ID
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     bool                        returns true on success
	 */
	public function closeSession( $connectId );


	/**
	 * Get zanox User Interface Url
	 *
	 * @param      string $connectId connect ID
	 * @param      string $sessionKey session key
	 *
	 * @access     public
	 * @category   signature
	 *
	 * @return     bool                        returns true on success
	 */
	public function getUiUrl( $connectId, $sessionKey );


}

?>