<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

if ( ! defined( 'ATKP_VERSION_PRODUCTOFFERSTABLE' ) ) {
	define( 'ATKP_VERSION_PRODUCTOFFERSTABLE', 1 );
}

class atkp_offertable_helper {
	public function __construct() {
		$this->check_table_structure();

	}

	/**
	 * Gibt den internen Tabellenname inkl. Prefix für _offertable zurück
	 * @return string Der Tabellenname
	 */
	function get_offertable_tablename() {
		global $wpdb;

		return strtolower( $wpdb->prefix . ATKP_PLUGIN_PREFIX . '_productoffers' );;
	}

	function get_old_offertable_tablename() {
		global $wpdb;

		return strtolower( $wpdb->prefix . ATKP_PLUGIN_PREFIX . '_additionaloffers' );;
	}

	public function exists_table() {
		global $wpdb;
		$tablename = $this->get_offertable_tablename();
		$sql = "SHOW TABLES LIKE '".$tablename."'";

		$result = $wpdb->get_results( $sql );
		return array(count($result) > 0,  $tablename);
	}

	function add_offers_by_listid( $listid, $asin, $alloffers ) {

		$offers = array();
		foreach ( $alloffers as $offer ) {
			$offer->productid = 0;
			$offer->listid    = $listid;
			$offer->asin      = $asin;

			$offers[] = $offer;
		}
		$this->add_offers( $offers );
	}

	/**
	 * Aktualisiert oder hängt die übergebenen atkp_product_offer Angebote an der Tabelle an
	 *
	 * @param int $listid Die Listen-ID für die angehängten Angebote
	 * @param string $asin Die eindeutige Produkt-ID in der Liste
	 * @param Array $alloffers Ein Array von atkp_product_offer welche gespeichert werden soll
	 *
	 * @deprecated
	 */
	function update_offers_by_listid( $listid, $asin, $alloffers ) {
		global $wpdb;

		foreach ( $alloffers as $offer ) {
			$offer->productid = 0;
			$offer->listid    = $listid;
			$offer->asin      = $asin;

			$this->update_offer( $offer );
		}

	}

	function add_offers_by_productid( $productid, $alloffers ) {

		$offers = array();
		foreach ( $alloffers as $offer ) {
			$offer->productid = $productid;
			$offer->listid    = 0;
			$offer->asin      = '';

			$offers[] = $offer;
		}
		$this->add_offers( $offers );
	}

	function delete_offers_by_productid( $productid, $alloffers ) {
		$this->delete_offers( $alloffers, '', $productid );
	}

	function reset_offers_by_productid( $productid, $alloffers ) {
		$this->reset_offers( $alloffers, '', $productid );
	}


	/**
	 * Löscht alle Angebote zu diesem Produkt und hängt die übergebenen atkp_product_offer Angebote an
	 *
	 * @param int $productid Die Produkt-ID für die angehängten Angebote
	 * @param Array $alloffers Ein Array von atkp_product_offer welche gespeichert werden soll
	 * @param bool $clearoffers Gibt an ob die existierenden Angebote vor dem Speichern gelöscht werden sollen
	 *
	 * @deprecated
	 */
	function update_offers_by_productid( $productid, $alloffers, $clearoffers = true ) {
		//global $wpdb;

		//if($clearoffers)
		//	$this->clear_offers_internal( $productid, 0 );

		foreach ( $alloffers as $offer ) {
			$offer->productid = $productid;
			$offer->listid    = 0;
			$offer->asin      = '';

			$this->update_offer( $offer );
		}

	}


	function delete_offers_by_listid( $listid, $alloffers ) {
		$this->delete_offers( $alloffers, $listid, '' );
	}

	function reset_offers_by_listid( $listid, $alloffers ) {
		$this->reset_offers( $alloffers, $listid, '' );
	}

	/**
	 * Lädt alle Angebote welche zu der Liste + eindeutige Product-ID gehören
	 *
	 * @param int $listid Die Listen-ID
	 * @param string $asin Die eindeutige ID vom Listeneintrag
	 *
	 * @return array Gibt ein Array von atkp_product_offer zurück
	 */
	function get_offers_by_listid( $listid, $asin ) {
		return $this->get_offers_internal( '', $listid, $asin );
	}


	/**
	 * Lädt alle Angebote welche zu dieser AT-Produkt-ID gehören
	 *
	 * @param int $productid Die AT-Produkt-ID
	 *
	 * @return array Gibt ein Array von atkp_product_offer zurück
	 */
	function get_offers_by_productid( $productid ) {

		return $this->get_offers_internal( $productid, '', '' );
	}


	/**
	 * Prüft ob die Tabelle atkp_offertable vorhanden ist, und legt diese ggf. an.
	 */
	function check_table_structure() {
		global $wpdb;

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' ); //required for dbDelta

		$current_version_db = get_option( ATKP_PLUGIN_PREFIX . '_version_additionalofferstable_v2' );
		//$current_version_db = 0;
		$table_name = $this->get_offertable_tablename();

		//older or not even set
		if ( $current_version_db < ATKP_VERSION_PRODUCTOFFERSTABLE || $current_version_db == false ) {

			$sql = "CREATE TABLE $table_name (
				                    `Id` bigint (50) NOT NULL AUTO_INCREMENT,
				                    `ShopId` bigint (50) NOT NULL,
				                    `ProductId` bigint( 50 ) NOT NULL,
				                    `ListId` bigint( 50 ) NOT NULL,
				                    `ProductNumber` varchar( 250 ) NOT NULL,
				                    `ProductTitle` varchar(700) NULL ,
				                    `ProductEAN` varchar( 100 ) NULL ,
				                    `Shipping` varchar( 150 ) NULL ,
				                    `ShippingFloat` float( 10,2 ) NULL ,
				                    `Availability` varchar( 150 ) NULL ,
				                    `ProductPrice` varchar( 60 ) NULL ,
				                    `ProductPriceFloat` float( 10,2 ) NULL ,
				                    `OfferType` int(1) NULL ,
				                    `ProductLink` varchar(700) NULL ,
				                    `HideOffer` int(1) DEFAULT 0 ,
				                    `HoldOnTop` int(1) DEFAULT 0 ,
				                    `Message` varchar(700) NULL ,
				                    `UpdatedOn` datetime NULL ,
				                    `HasError` int(1) DEFAULT 0, 	        
				                     PRIMARY KEY (`Id`)				                     
				                    ) ENGINE = MYISAM DEFAULT CHARSET = utf8";

			dbDelta( $sql );

			$wpdb->query( "ALTER TABLE $table_name ADD INDEX atkp_idx_listid_productnumber ( `ListId`, `ProductNumber`);" );
			$wpdb->query( "ALTER TABLE $table_name ADD INDEX atkp_idx_productid ( `ProductId`);" );
			$wpdb->query( "ALTER TABLE $table_name ADD INDEX atkp_idx_productid_offertype_updatedon ( `shopid`,`offertype`,`UpdatedOn`);" );

			$wpdb->query( "ALTER TABLE $table_name ADD INDEX atkp_idx_productid_get ( `ShopId`,`OfferType`,`ProductId`,`UpdatedOn`,`HasError`);" );
			$wpdb->query( "ALTER TABLE $table_name ADD INDEX atkp_idx_listid_get ( `ShopId`,`OfferType`,`ListId`,`UpdatedOn`,`HasError`);" );

			update_option( ATKP_PLUGIN_PREFIX . '_version_additionalofferstable_v2', ATKP_VERSION_PRODUCTOFFERSTABLE );

			//delete old table:
			$oldtablename = $this->get_old_offertable_tablename();
			//if($oldtablename != '')
			//	$wpdb->query( "DROP TABLE IF EXISTS $oldtablename;" );

		}
	}

	private function check_exists( $offerId ) {
		global $wpdb;
		$table_name = $this->get_offertable_tablename();

		$rowcount = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name WHERE Id = '$offerId' " );

		return $rowcount;
	}

	private function get_offer_save_array( $offer, $initial_insert = false ) {
		$data = array(
			'ProductId'         => intval( $offer->productid ),
			'ListId'            => intval( $offer->listid ),
			'ProductTitle'      => $offer->title,
			'ProductNumber'     => $offer->asin,
			'Shipping'          => $offer->shipping,
			'ShippingFloat'     => $offer->shippingfloat,
			'Availability'      => $offer->availability,
			'ProductPrice'      => $offer->price,
			'ProductPriceFloat' => $offer->pricefloat,
			'ShopId'            => intval( $offer->shopid ),
			'OfferType'         => intval( $offer->type ),
			'ProductEAN'        => $offer->number,
			'ProductLink'       => $offer->link,
			'HideOffer'         => $offer->hideoffer ? 1 : 0,
			'HoldOnTop'         => $offer->holdontop ? 1 : 0,
			'Message'           => $offer->message == '' ? null : $offer->message,
			'HasError'          => $offer->message == '' ? 0 : 1,
			'UpdatedOn'         => $initial_insert ? null : date( "Y-m-d H:i:s" ),
		);

		if ( ! $initial_insert ) {
			$data['Id'] = intval( $offer->id );
		}

		$data = apply_filters( 'atkp_modify_offer_before_db_write', $data );

		return $data;
	}

	private function add_offers( $offers ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();
		$updated    = array();
		foreach ( $offers as $offer ) {

			$data = $this->get_offer_save_array( $offer, true );

			$wpdb->insert(
				$table_name,
				$data
			);

			$offer->id = $wpdb->insert_id;
			array_push( $updated, $offer );
		}

		return $updated;
	}

	private function reset_offers( $offers, $listid, $productid ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$ids = array();
		foreach ( $offers as $offer ) {
			$ids[] = $offer->id;
		}

		$ids2 = implode( ',', array_map( 'absint', $ids ) );
		if ( $listid != '' ) {
			$affected = $wpdb->query( "update $table_name set haserror=0 WHERE Id IN($ids2)" );
		} else {
			$affected = $wpdb->query( "update $table_name set haserror=0 WHERE Id IN($ids2)" );
		}


		return $affected;
	}

	private function delete_offers( $offers, $listid, $productid ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$ids = array();
		foreach ( $offers as $offer ) {
			$ids[] = $offer->id;
		}

		$ids2 = implode( ',', array_map( 'absint', $ids ) );
		if ( $listid != '' ) {
			$affected = $wpdb->query( "DELETE FROM $table_name WHERE Id IN($ids2)" );
		} else {
			$affected = $wpdb->query( "DELETE FROM $table_name WHERE Id IN($ids2)" );
		}


		return $affected;
	}

	/**
	 * Interne Funktion für das Angebote aktualisieren
	 *
	 * @param $productid
	 * @param $listid
	 * @param $productnumber
	 * @param $offer
	 *
	 * @return false|int
	 */
	public function update_offer( $offer ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();


		$where = array(
			'Id' => intval( $offer->id ),
		);

		//insert oder update row
		if ( $this->check_exists( intval( $offer->id ) ) ) {

			$data = $this->get_offer_save_array( $offer, false );

			$updated = $wpdb->update(
				$table_name,
				$data,
				$where
			);

		} else {
			$data = $this->get_offer_save_array( $offer, true );

			$updated   = $wpdb->insert(
				$table_name,
				$data
			);
			$offer->id = $wpdb->insert_id;
		}

		return $offer;
	}


	/**
	 * Interne Funktion welche von den beiden externen Funktionen aufgerufen wird.
	 *
	 * @param $productid
	 * @param $listid
	 * @param $asin
	 *
	 * @return array
	 */
	private function get_offers_internal( $productid, $listid, $asin ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		if ( $listid != '' ) {
			$result = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name WHERE listid = %s and productnumber = %s	", $listid, $asin ), ARRAY_A );
		} else {
			$result = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name WHERE productid = %s	", $productid ), ARRAY_A );
		}

		return $this->build_offer_array( $result );
	}

	public function get_productids() {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$result = $wpdb->get_results( "SELECT productid FROM $table_name group by productid	", ARRAY_A );

		$resultxx = array();

		foreach ( $result as $xx ) {
			array_push( $resultxx, $xx['productid'] );
		}

		return $resultxx;
	}

	public function delete_productids( $productids ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$ids = implode( ',', array_map( 'absint', $productids ) );
		$wpdb->query( "DELETE FROM $table_name WHERE productid IN($ids)" );
	}

	public function delete_listids( $listids ) {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$ids = implode( ',', array_map( 'absint', $listids ) );
		$wpdb->query( "DELETE FROM $table_name WHERE listid IN($ids)" );
	}

	public function get_listids() {
		global $wpdb;

		$table_name = $this->get_offertable_tablename();

		$result = $wpdb->get_results( "SELECT listid FROM $table_name group by listid	", ARRAY_A );

		$resultxx = array();

		foreach ( $result as $xx ) {
			array_push( $resultxx, $xx['listid'] );
		}

		return $resultxx;
	}

	public function get_offers_for_update( $shopid, $offertype, $page = 1, $productid = 0, $updatetime = 0, $items_per_page = 10 ) {
		global $wpdb;

		$offset = ( $page * $items_per_page ) - $items_per_page;

		$table_name = $this->get_offertable_tablename();


		$query = '';

		if ( $productid > 0 ) {

			switch ( $offertype ) {
				case 'ASIN':
					$query = $wpdb->prepare( "SELECT * FROM $table_name WHERE shopid = %s and productid= %s and offertype = %s", $shopid, $productid, atkp_product_offer::OFFER_TYPE_PRODUCTID );
					break;
				case 'EAN':
					$query = $wpdb->prepare( "SELECT * FROM $table_name WHERE shopid = %s and productid = %s and offertype = %s", $shopid, $productid, atkp_product_offer::OFFER_TXPE_EAN );
					break;
				default:
					throw new Exception( 'unknown offertype: ' . $offertype );
			}
		} else {
			$haserrorsql = '';

			if ( atkp_options::$loader->get_ignoreoffernotfound() ) {
				$haserrorsql = 'and haserror = 0';
			}

			switch ( $offertype ) {
				case 'ASIN':
					$query = $wpdb->prepare( "SELECT * FROM $table_name WHERE shopid = %s and offertype = %s and (UpdatedOn is null OR UNIX_TIMESTAMP(UpdatedOn)  < %d ) " . $haserrorsql, $shopid, atkp_product_offer::OFFER_TYPE_PRODUCTID, $updatetime );
					break;
				case 'EAN':
					$query = $wpdb->prepare( "SELECT * FROM $table_name WHERE shopid = %s and offertype = %s and (UpdatedOn is null OR UNIX_TIMESTAMP(UpdatedOn)  < %d ) " . $haserrorsql, $shopid, atkp_product_offer::OFFER_TXPE_EAN, $updatetime );
					break;
				default:
					throw new Exception( 'unknown offertype: ' . $offertype );
			}
		}

		$result = $wpdb->get_results( $query . " ORDER BY Id LIMIT ${offset}, ${items_per_page}", ARRAY_A );

		return $this->build_offer_array( $result );
	}

	private function build_offer_array( $result ) {

		$products = array();

		if ( $result ) {
			foreach ( $result as $row ) {

				$product = new atkp_product_offer();

				$product->productid     = $row['ProductId'];
				$product->listid        = $row['ListId'];
				$product->asin          = $row['ProductNumber'];
				$product->title         = $row['ProductTitle'];
				$product->id            = $row['Id'];
				$product->shipping      = $row['Shipping'];
				$product->shippingfloat = $row['ShippingFloat'];
				$product->availability  = $row['Availability'];
				$product->price         = $row['ProductPrice'];
				$product->pricefloat    = $row['ProductPriceFloat'];
				$product->shopid        = $row['ShopId'];
				$product->type          = $row['OfferType'];
				$product->number        = $row['ProductEAN'];
				$product->link          = $row['ProductLink'];
				$product->hideoffer     = $row['HideOffer'] == 1;
				$product->holdontop     = $row['HoldOnTop'] == 1;
				$product->message       = $row['Message'];
				$product->updatedon     = $row['UpdatedOn'];

				array_push( $products, $product );
			}
		}

		return $products;

	}

}